/**
  @file             SM_LLL.c
  @version          0.1.0.0          

  @brief            
  @details          

  Project           Stepper Motor Driver
  Platform          MPC5606S

  SWVersion         0.1.0


  (c) Copyright 2011 Freescale Semiconductor Inc.
  All Rights Reserved.
*/
/*==================================================================================================
==================================================================================================*/

#ifdef __cplusplus
extern "C"{
#endif

/*==================================================================================================
                                         INCLUDE FILES
==================================================================================================*/


#include "MAC57D54H.h"
#include "SM_Tables.h"
#include "SM_Global.h"
#include "SM_LLL.h"
#include "SM_RAL.H"

#include "arm_cm4.h"

/*==================================================================================================
                                 STRUCTURES AND OTHER TYPEDEFS
==================================================================================================*/

typedef struct 				/* Structure contains data to transfer from the SM_LLL_DrivePwms function 
								to the SMC overflow interrupt */ 
{
	SM_MotorHwInitState_t	eMotorHwInit;		/* Hardware initialization state */ 
	uint16_t 				u16MotorSin;		/* Sin coil PWM value */
	uint16_t 				u16MotorCos;		/* Cos coil PWM value */
}LLL_SmcData_t;

/*==================================================================================================
                                       GLOBAL VARIABLES
==================================================================================================*/

LLL_SmcData_t		trSmcData[SM_MAX_MOTORS];	/* Array of structures to transfer data into 
													the SMC overflow interrupt */ 

/*==================================================================================================
                                       LOCAL CONSTANTS
==================================================================================================*/

#ifdef SM_TABLE_128								/* SM Sin/Cos table of 128 elements */	
    const uint8_t  LLL_SinCosTbl [128] = 
    {
    	0,13,25,37,50,62,74,86,98,109,120,131,142,152,162,171,180,189,197,205,212,219,225,231,236,
    	240,244,247,250,252,254,255,255,255,254,252,250,247,244,240,236,231,225,219,212,205,197,189,
    	180,171,162,152,142,131,120,109,98,86,74,62,50,37,25,13,0,13,25,37,50,62,74,86,98,109,120,131,
    	142,152,162,171,180,189,197,205,212,219,225,231,236,240,244,247,250,252,254,255,255,255,254,
    	252,250,247,244,240,236,231,225,219,212,205,197,189,180,171,162,152,142,131,120,109,98,86,74,
    	62,50,37,25,13
    };
#endif

#ifdef SM_TABLE_256								/* SM Sin/Cos table of 256 elements */
    const uint8_t  LLL_SinCosTbl [256] = 	
    {
    	0,6,13,19,25,31,37,44,50,56,62,68,74,80,86,92,98,103,109,115,120,126,131,136,142,147,152,157,
    	162,167,171,176,180,185,189,193,197,201,205,208,212,215,219,222,225,228,231,233,236,238,240,
    	242,244,246,247,249,250,251,252,253,254,254,255,255,255,255,255,254,254,253,252,251,250,249,
    	247,246,244,242,240,238,236,233,231,228,225,222,219,215,212,208,205,201,197,193,189,185,180,
    	176,171,167,162,157,152,147,142,136,131,126,120,115,109,103,98,92,86,80,74,68,62,56,50,44,37,
    	31,25,19,13,6,0,6,13,19,25,31,37,44,50,56,62,68,74,80,86,92,98,103,109,115,120,126,131,136,
    	142,147,152,157,162,167,171,176,180,185,189,193,197,201,205,208,212,215,219,222,225,228,231,
    	233,236,238,240,242,244,246,247,249,250,251,252,253,254,254,255,255,255,255,255,254,254,253,
    	252,251,250,249,247,246,244,242,240,238,236,233,231,228,225,222,219,215,212,208,205,201,197,
    	193,189,185,180,176,171,167,162,157,152,147,142,136,131,126,120,115,109,103,98,92,86,80,74,68,
    	62,56,50,44,37,31,25,19,13,6
    };
#endif

/*==================================================================================================
                                       GLOBAL FUNCTIONS
==================================================================================================*/

/*******************************************************************************
* @function_name SM_LLL_DrivePwms
*//*
* @brief 		The function drives PWM channels of the corresponding motor to move 
				it into a new position  
* @param[in] 	prMtr -	structure containing all the motor parameters 
* @details 		When the motor hardware is initialized, the function picks the 
*				element of the Sin/Cos table according to actual value of the 
*				"u8Step" variable in prMtr.trMtrAlgo structure. This element is
*				stored as a Sin coil value and the table index is turned by 
*				90 degrees.	Another table element is picked and stored as a Cos
*				coil value. At the end of the function the SMC overflow interrupt
*				is enabled to synchronous update of the PWM values. 
*/

void SM_LLL_DrivePwms(SM_MotorParams_t *prMtr)
{
	const SM_Table_t *prTable;
	
	uint16_t  u16PwmValCoilSin,u16PwmValCoilCos;
	uint16_t  u16TblIndex;
	uint8_t	  u8Quad;
 
 	prTable = prMtr->trGeneral.cpTableSet;
 
	/* Only if the motor HW is initialized */ 
	if(prMtr->trGeneral.eMotorHwInitialized == SM_HW_MTR_INITIALIZED)
	{ 	    
	    u16TblIndex = prMtr->trMtrAlgo.u16Step;		/* Table index */
	   
	     
		u16PwmValCoilSin = prTable->Speed_Dir_Index[prMtr->trMtrAlgo.u8TblSpeed][prMtr->trMtrAlgo.eMtrCmdDir][u16TblIndex].u10Sin;
		u16PwmValCoilCos = prTable->Speed_Dir_Index[prMtr->trMtrAlgo.u8TblSpeed][prMtr->trMtrAlgo.eMtrCmdDir][u16TblIndex].u10Cos;
		u8Quad = prTable->Speed_Dir_Index[prMtr->trMtrAlgo.u8TblSpeed][prMtr->trMtrAlgo.eMtrCmdDir][u16TblIndex].u2Quad;	    	
    			
		switch(u8Quad)
		{
			case 1:
				u16PwmValCoilSin &= 0x7FFF;
				u16PwmValCoilCos &= 0x7FFF;
			break;
			
			case 0:
				u16PwmValCoilSin |= 0x8000;
				u16PwmValCoilCos &= 0x7FFF;
			break;
			
			case 2:
				u16PwmValCoilSin &= 0x7FFF;
				u16PwmValCoilCos |= 0x8000;
			break;
				
			case 3:
				u16PwmValCoilSin |= 0x8000;
				u16PwmValCoilCos |= 0x8000;
			break;
			
			default:
			break;
		}
		
	
		/* Store coil data into a SmcData structure that transfers data into a SMC overflow 
			interrupt hendler*/
  		trSmcData[prMtr->trGeneral.eMotorId].u16MotorSin = u16PwmValCoilSin;
	  	trSmcData[prMtr->trGeneral.eMotorId].u16MotorCos = u16PwmValCoilCos;
	  	trSmcData[prMtr->trGeneral.eMotorId].eMotorHwInit = prMtr->trGeneral.eMotorHwInitialized;
	  	
	  	/* Clear SMC overflow interrupt flag */
	  	LLL_SMC_OVF_INT_CLEAR;
	  	/* Enable SMC overflow interrupt. Once the SMC PWM timer overflows, interrupt is triggered */
	  	LLL_SMC_OVF_INT_ENABLE;
	}
}
	
/*******************************************************************************
* @function_name SM_LLL_Rtz
*//*
* @brief 		The function operates RTZ movement  
* @param[in] 	prMtr -	structure containing all the motor parameters
* @details 		When the motor hardware is initialized the SSD step is updated 
*				according to the required stop direction. At the end, the 
*				blanking-integration sequence is triggered. 
*/

void SM_LLL_Rtz(SM_MotorParams_t *prMtr)
{
	volatile struct SSD_tag* prSsdX;

	/* Only if the motor HW is initialized */
	if(prMtr->trGeneral.eMotorHwInitialized == SM_HW_MTR_INITIALIZED)
	{
		/* Determine which SSD module is going to be controlled */
		switch(prMtr->trGeneral.eMotorId)
		{	
			case SM_MOTOR_ID_1:
				prSsdX = (&SSD_0);		/* Copy the address of the SSD0 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_2:
				prSsdX = (&SSD_1);		/* Copy the address of the SSD1 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_3:
				prSsdX = (&SSD_2);		/* Copy the address of the SSD2 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_4:
				prSsdX = (&SSD_3);		/* Copy the address of the SSD3 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_5:
				prSsdX = (&SSD_4);		/* Copy the address of the SSD4 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_6:
				prSsdX = (&SSD_5);		/* Copy the address of the SSD5 into the temporary pointer */
			break;
			
			default:
			break;
		}				
			
		LLL_SSDX_STEP_DEC(prSsdX);
			 	
		/* Trigger the blanking-integration sequence that measures back EMF on the undriven coil */
		LLL_SSDX_BIS_TRIG(prSsdX);
	}
}

/*******************************************************************************
* @function_name SM_LLL_InitHwModules
*//*
* @brief 		The function initializes HW resources for stepper motor driver 
* @param[in] 	prMtr -	structure containing all the motor parameters
* @details 		Function initializes SMC module ... TBD (different on the HALO) 
*/

void SM_LLL_InitHwModules(SM_MotorParams_t *prMtr)
{
	/* SMC_MCCTL0 :
	 	SMC_MCCTL0[2] (DITH) = 1;		- Dither Feature Enabled
		SMC_MCCTL0[5:6] (MCPRE) = 3   	- Prescaller, fTC = BUS_CLK / 8 */	 
	RAL_RegWrite(0x64u,&RAL_SMC_REG_MCCTL0);
	
	/* SMC_MCCTL1 :
		Recirculation on the High side transistors 
		Timer Counter Overflow Interrupt disabled */
	RAL_RegWrite(0x00u,&RAL_SMC_REG_MCCTL1);
	
	/* SMC_MCPER :
		PWM period - 0xFF */		
	RAL_RegWrite(0x3FFu,&RAL_SMC_REG_MCPER );
		
	/* SMC_MCCC :
		SMC_MCCC[4:5] (MCAM) = 3;		- PWM Channel Alignment mode - Center aligned
		SMC_MCCC[6:7] (MCOM) = 3;		- Output Mode - Dual full H bridge mode */
	RAL_RegWrite(0xF0u,&RAL_MOTOR_CC_COS(prMtr->trGeneral.eMotorId));
	RAL_RegWrite(0xF0u,&RAL_MOTOR_CC_SIN(prMtr->trGeneral.eMotorId));
	
	/* SMC_DUTY - Actual duty cycle is 0 */
	RAL_RegWrite(0x00u,&RAL_MOTOR_DUTY_COS(prMtr->trGeneral.eMotorId));
	RAL_RegWrite(0x00u,&RAL_MOTOR_DUTY_SIN(prMtr->trGeneral.eMotorId));
	
	/* Connect SMC module to the stepper motor pins */
	switch (prMtr->trGeneral.eMotorId){
		case SM_MOTOR_ID_1:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_0_OFFSET,LLL_SIUL2_SMC);
			break;
		case SM_MOTOR_ID_2:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_1_OFFSET,LLL_SIUL2_SMC);
			break;
		case SM_MOTOR_ID_3:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_2_OFFSET,LLL_SIUL2_SMC);
			break;
		case SM_MOTOR_ID_4:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_3_OFFSET,LLL_SIUL2_SMC);
			break;
		case SM_MOTOR_ID_5:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_4_OFFSET,LLL_SIUL2_SMC);
			break;
		case SM_MOTOR_ID_6:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_5_OFFSET,LLL_SIUL2_SMC);
			break;
		default:
			break;
	}
	//LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR(prMtr->trGeneral.eMotorId),LLL_SIUL2_SMC);         
	
	/* next row doesnt follow code structure - RAL access because will be modified in final application (Halo) */
	//INTC.PSR[192].R = 1u;		      /* Motor Controller Timer Counter Overflow Interrupt IRQ priority = 1  */
	route_interrupt(20,1);       
	set_irq_priority(20,7);
	enable_irq(20); //SMC
	
	/* Mark the corresponding stepper motor HW as INITIALIZED */
	prMtr->trGeneral.eMotorHwInitialized = SM_HW_MTR_INITIALIZED;
}

/*******************************************************************************
* @function_name SM_LLL_SsdInit
*//*
* @brief 		The function initializes SSD module.  
* @param[in] 	prMtr -	structure containing all the motor parameters
* @details 		When the motor hardware is initialized, output of the corresponding 
*				SSD module is activated and the SSD interrupt priority is set. Than 
*				all the user defined parameters are used to set SSD module registers 
*				and the SSD Integration Expired interrupt is enabled.
*/

void SM_LLL_SsdInit(SM_MotorParams_t *prMtr)
{
	uint16_t u16ControlByte, u16Step;
	volatile struct SSD_tag* prSsdX;

	/* Only if the motor hardware is initialized */
	if(prMtr->trGeneral.eMotorHwInitialized == SM_HW_MTR_INITIALIZED)
	{			
		/* Determine which SSD module is going to be initialized */
		switch(prMtr->trGeneral.eMotorId)
		{
			case SM_MOTOR_ID_1:
				/* Copy the address of the SSD0 into the temporary pointer */
				prSsdX = (&SSD_0);			
				//INTC.PSR[193].R = 0x01u;		/* PIT 0 interrupt vector with priority 1 */
				route_interrupt(21,1);        /* Route interrupt to primary CPU */
				set_irq_priority(21,7);
				enable_irq(21); //SSD_0
				/* Connect motor 1 outputs to the SSD0 module */
				LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_0_OFFSET,LLL_SIUL2_SSD);         
			break;
			
			case SM_MOTOR_ID_2:
				/* Copy the address of the SSD1 into the temporary pointer */
				prSsdX = (&SSD_1);
				//INTC.PSR[194].R = 0x01u;        /* PIT 0 interrupt vector with priority 1 */
				route_interrupt(22,1);        /* Route interrupt to primary CPU */
				set_irq_priority(22,7);
				enable_irq(22); //SSD_1
				/* Connect motor 2 outputs to the SSD1 module */
				LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_1_OFFSET,LLL_SIUL2_SSD); 
			break;
			
			case SM_MOTOR_ID_3:
				/* Copy the address of the SSD2 into the temporary pointer */
				prSsdX = (&SSD_2);
				//INTC.PSR[195].R = 0x01u;        /* PIT 0 interrupt vector with priority 1 */
				route_interrupt(23,1);        /* Route interrupt to primary CPU */
				set_irq_priority(23,7);
				enable_irq(23); //SSD_2
				/* Connect motor 3 outputs to the SSD2 module */
				LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_2_OFFSET,LLL_SIUL2_SSD); 
			break;
			
			case SM_MOTOR_ID_4:
				/* Copy the address of the SSD3 into the temporary pointer */
				prSsdX = (&SSD_3);	
				//INTC.PSR[196].R = 0x01u;        /* PIT 0 interrupt vector with priority 1 */
				route_interrupt(24,1);        /* Route interrupt to primary CPU */
				set_irq_priority(24,7);
				enable_irq(24); //SSD_3
				/* Connect motor 4 outputs to the SSD3 module */
				LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_3_OFFSET,LLL_SIUL2_SSD); 
			break;
			
			case SM_MOTOR_ID_5:
				/* Copy the address of the SSD4 into the temporary pointer */
				prSsdX = (&SSD_4);		
				//INTC.PSR[197].R = 0x01u;        /* PIT 0 interrupt vector with priority 1 */
				route_interrupt(25,1);        /* Route interrupt to primary CPU */
				set_irq_priority(25,7);
				enable_irq(25); //SSD_4
				/* Connect motor 5 outputs to the SSD4 module */
				LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_4_OFFSET,LLL_SIUL2_SSD); 
			break;
			
			case SM_MOTOR_ID_6:
				/* Copy the address of the SSD5 into the temporary pointer */
				prSsdX = (&SSD_5);		
				//INTC.PSR[198].R = 0x01u;        /* PIT 0 interrupt vector with priority 1 */
				route_interrupt(26,1);        /* Route interrupt to primary CPU */
				set_irq_priority(26,7);
				enable_irq(26); //SSD_5
				/* Connect motor 6 outputs to the SSD5 module */
				LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_5_OFFSET,LLL_SIUL2_SSD); 
			break;
			
			default:
			break;
		}
				
		/* "u16ControlByte" corresponds with SSD_CONTROL register */
		/* SSD_CONTROL[10] (BLNDCL) = 1;	- coil is driven during the Blanking-Integration Sequence (BIS) */
		/* SSD_CONTROL[9] (ITGDCL) = 1;		- coil is driven also outcide of any BIS */
		/* SSD_CONTROL[8] (RTZE) = 1;		- Return to zero ENABLE */
		/* SSD_CONTROL[1] (SDCPU) = 1;		- Sigma-Delta modulator Power Up */
		/* SSD_CONTROL[13:14] (STEP) = 		- Required start quadrant*/ 
		
		u16ControlByte = 0x0702u | (prMtr->trRtz.eRtzStartQuad << 13);
			   		
		/* If the motor stop direction is clockwise */
		//if(prMtr->trGeneral.eMotorStopDirection == SM_STOP_DIRECTION_CW)
		{
			/* SSD_CONTROL[11] (ITGDIR) = 1;	- Polarity of the integration set accordingly */
			u16ControlByte |= 0x0800u;
		}
		
		/* Assign calculated integration time into the SSD_ITGCNTLD register */ 	
		RAL_RegWrite(prMtr->trRtz.u16RtzIntegrationTime,&prSsdX->RAL_SSDX_REG_ITGCNTLD);
		
		/* Assigne user defined blanking time into the SSD_BLNCNTLD register */
		RAL_RegWrite(prMtr->trRtz.u16RtzBlankingTime,&prSsdX->RAL_SSDX_REG_BLNCNTLD);
		
		/* Set SSD prescalers: 
			SSD_PRESCALE[0:2] (ACDIV) =	3;	- Accumulator Sample Clock = BUS_CLK / 64 
			SSD_PRESCALE[4:5] (OFFCNC = 3;	- Offset Cancelation Polarity Flip = ITGCNTLD / 8
			SSD_PRESCALE[8:10](ITGDIV) = 3;	- Integration Counter Clock Divider = BUS_CLK / 64
			SSD_PRESCALE[12:14](BLNDIV) = 3;- Blanking Counter Clock Divider = BUS_CLK / 64 */
		RAL_RegWrite(0x3333u,&prSsdX->RAL_SSDX_REG_PRESCALE);
		
		/* Assigne "u16ControlByte" into the SSD_CONTROL register */
		RAL_RegWrite(u16ControlByte,&prSsdX->RAL_SSDX_REG_CONTROL);	
		
		/* Enable SSD Integration Expired interrupt */
		LLL_SSDX_INT_ITG_ENABLE(prSsdX);
	}
}

/*******************************************************************************
* @function_name SM_LLL_DisableHwModules
*//*
* @brief 		The function disables HW resources for stepper motor driver 
* @param[in] 	prMtr -	structure containing all the motor parameters
* @details 		Function clears all the PWM registers of the SMC module and turns
*				stepper motor outputs into a GPIO outputs  
*/

void SM_LLL_DisableHwModules(SM_MotorParams_t *prMtr)
{
	/* Mark the corresponding stepper motor HW as DISABLED */
	prMtr->trGeneral.eMotorHwInitialized = SM_HW_MTR_DISABLED;
	
	/* Clear Cos and Sin SMC Channel Control Registers */  
	RAL_RegWrite(0x00u,&RAL_MOTOR_CC_COS(prMtr->trGeneral.eMotorId));
	RAL_RegWrite(0x00u,&RAL_MOTOR_CC_SIN(prMtr->trGeneral.eMotorId));
	
	/* Clear Cos and Sin SMC Duty Cycle Registers */
	RAL_RegWrite(0x00u,&RAL_MOTOR_DUTY_COS(prMtr->trGeneral.eMotorId));
	RAL_RegWrite(0x00u,&RAL_MOTOR_DUTY_SIN(prMtr->trGeneral.eMotorId));
	
	/* Disconnect SMC module from the stepper motor pins */
	switch (prMtr->trGeneral.eMotorId){
		case SM_MOTOR_ID_1:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_0_OFFSET,LLL_SIUL2_GPIO);
			break;
		case SM_MOTOR_ID_2:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_1_OFFSET,LLL_SIUL2_GPIO);
			break;
		case SM_MOTOR_ID_3:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_2_OFFSET,LLL_SIUL2_GPIO);
			break;
		case SM_MOTOR_ID_4:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_3_OFFSET,LLL_SIUL2_GPIO);
			break;
		case SM_MOTOR_ID_5:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_4_OFFSET,LLL_SIUL2_GPIO);
			break;
		case SM_MOTOR_ID_6:
			LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_5_OFFSET,LLL_SIUL2_GPIO);
			break;
		default:
			break;
	}
	//LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR(prMtr->trGeneral.eMotorId),LLL_SIUL2_GPIO); 
}

/*******************************************************************************
* @function_name SM_LLL_SsdInterrupt
*//*
* @brief 		The function services SSD Integration Expired interrupt	
* @param[in] 	prMtr -	structure containing all the motor parameters
* @details 		When the motor hardware is initialized and the result of the 
*				integration is valid, the result of the integration is compared 
*				with value of "s16RtzBemfThreshold" in the "prMtr.trRtz" structure.
*	 			If result of the integration is lower than defined threshold, 
*				the RTZ is done and actual motor position is defined as 0.   	
*/

void SM_LLL_SsdInterrupt(SM_MotorParams_t *prMtr)
{
	volatile struct SSD_tag* prSsdX;
	int16_t s16IntegrationResult;
	uint16_t u16StopQuad;

	/* Only if the motor hardware is initialized */
	if(prMtr->trGeneral.eMotorHwInitialized == SM_HW_MTR_INITIALIZED)
	{	
		/* Determine which SSD module is going to be controlled */
		switch(prMtr->trGeneral.eMotorId)
		{
		
			case SM_MOTOR_ID_1:
				prSsdX = (&SSD_0);		/* Copy the address of the SSD0 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_2:
				prSsdX = (&SSD_1);		/* Copy the address of the SSD1 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_3:
				prSsdX = (&SSD_2);		/* Copy the address of the SSD2 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_4:
				prSsdX = (&SSD_3);		/* Copy the address of the SSD3 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_5:
				prSsdX = (&SSD_4);		/* Copy the address of the SSD4 into the temporary pointer */
			break;
			
			case SM_MOTOR_ID_6:
				prSsdX = (&SSD_5);		/* Copy the address of the SSD5 into the temporary pointer */
			break;
			
			default:
			break;
		}
		
		/* If the result of the integration is valid (outside of the damping interval) */
		if(prMtr->trRtz.eIntResultStat == SM_HLL_INT_RESULT_VALID)
		{
			/* Load result of the integration from SSD ITGACC register into "s16IntegrationResult" */
			s16IntegrationResult = (int16_t) LLL_SSDX_ITGACC(prSsdX);
									
			/* if the integration result > back EMF threshold (stall has been detected) */				
			if(s16IntegrationResult > prMtr->trRtz.s16RtzBemfThreshold)
			{		
				prMtr->trMtrAlgo.s32Pos = 0;		/* Actual position defined as 0 */
				prMtr->trMtrAlgo.s16ActPos = 0u;
		    	prMtr->trMtrAlgo.s16TargetPos = 0u;
		    	prMtr->trMtrAlgo.s16TempCalc1 = 0u;
		    	prMtr->trMtrAlgo.s16ActSpeed = 0u;
		    	prMtr->trMtrAlgo.s16PrevSpeed = 0u;
		    	prMtr->trMtrAlgo.s16ActAcc = 0u;
		    	prMtr->trMtrAlgo.s16OldPos = 0u;
				
				prMtr->trGeneral.eZeroDetected = SM_ZERO_DETECTED;
				
				/* Load actual SSD STEP from the SSD_CONTROL refister */
				u16StopQuad = LLL_SSDX_ACTUAL_STEP(prSsdX);	
				
				/* Update actual algorithm step "u8Step" in prMtr.trMtrAlgo */ 
				prMtr->trRtz.eRtzStopQuad = u16StopQuad;
				
				/* Drive PWMs to settle rotor in actual position */
				//SM_LLL_DrivePwms(prMtr); 
				
				/* Disconnect motor outputs from SSD module and connect them to the SMC module */
				switch (prMtr->trGeneral.eMotorId){
					case SM_MOTOR_ID_1:
						LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_0_OFFSET,LLL_SIUL2_SMC);
						break;
					case SM_MOTOR_ID_2:
						LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_1_OFFSET,LLL_SIUL2_SMC);
						break;
					case SM_MOTOR_ID_3:
						LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_2_OFFSET,LLL_SIUL2_SMC);
						break;
					case SM_MOTOR_ID_4:
						LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_3_OFFSET,LLL_SIUL2_SMC);
						break;
					case SM_MOTOR_ID_5:
						LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_4_OFFSET,LLL_SIUL2_SMC);
						break;
					case SM_MOTOR_ID_6:
						LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR_5_OFFSET,LLL_SIUL2_SMC);
						break;
					default:
						break;
				}
				//LLL_SIUL2_OUTPUT(LLL_SIUL2_MOTOR(prMtr->trGeneral.eMotorId),LLL_SIUL2_SMC);
				
				/* Disable SSD Integration Expired interrupt */
				LLL_SSDX_INT_ITG_DISABLE(prSsdX);
				
				/* Update motor state to RTZ DONE */
				prMtr->trGeneral.eMotorState = SM_MOTOR_STATE_RTZ_DONE;
			}
		}
		
		else
		{
			prMtr->trRtz.eIntResultStat = SM_HLL_INT_RESULT_VALID;	
		}
		
		/* Clear SSD Integration Expired interrupt flag */
		LLL_SSDX_INT_CLEAR(prSsdX);
	}
}

/*******************************************************************************
* @function_name SM_LLL_TimerInit
*//*
* @brief 		The function initializes HW timer that controls SM driver 
* @param[in] 	eMtrId 	- motor timer going to be initialized 
* @details		tbd
* @note			The function will be modified on the HALO platform
*/

void SM_LLL_TimerInit(SM_MotorId_t eMtrId)
{
	switch (eMtrId){
	case SM_MOTOR_ID_1:
	case SM_MOTOR_ID_2:
	case SM_MOTOR_ID_3:
	case SM_MOTOR_ID_4:
		STM_0.CR.R = 0x00000702u; 										/* Prescaler = 8 STM @ 8MHz, Freeze enable */
	
		STM_0.CHANNEL[eMtrId].CCR.R = 0x00000001u; 							/* Channel 0 enabled */
		STM_0.CHANNEL[eMtrId].CIR.R = 0x00000001u; 							/* Channel 0 Interrupt enabled */
		STM_0.CHANNEL[eMtrId].CMP.R = STM_0.CHANNEL[eMtrId].CMP.R + SM_TIME_1MS;		/* Period: 1 ms */
		//INTC.PSR[eMtrId + 30u].R = 0x01u;								/* STM0 Channel 0 interrupt vector with priority 1 */ 
		route_interrupt(107,1);       
		set_irq_priority(107,7);
		enable_irq(107); //STM 0
	
		STM_0.CR.R = 0x00000703u; 										/* Enable Timer */
		break;
	case SM_MOTOR_ID_5:
	case SM_MOTOR_ID_6:
		STM_1.CR.R = 0x00000702u; 										/* Prescaler = 8 STM @ 8MHz, Freeze enable */
	
		STM_1.CHANNEL[eMtrId-4].CCR.R = 0x00000001u; 							/* Channel 0 enabled */
		STM_1.CHANNEL[eMtrId-4].CIR.R = 0x00000001u; 							/* Channel 0 Interrupt enabled */
		STM_1.CHANNEL[eMtrId-4].CMP.R = STM_1.CHANNEL[eMtrId-4].CMP.R + SM_TIME_1MS;		/* Period: 1 ms */
		//INTC.PSR[eMtrId + 30u].R = 0x01u;								/* STM0 Channel 0 interrupt vector with priority 1 */ 
		route_interrupt(100,1);       
		set_irq_priority(100,7);
		enable_irq(100); //STM 1
	
		STM_1.CR.R = 0x00000703u; 										/* Enable Timer */
		break;
	}
}

/*******************************************************************************
* @function_name SM_LLL_TimerDisable
*//*
* @brief 		The function disables HW timer that controls SM driver
* @param[in] 	eMtrId 	- motor timer going to be disabled 
* @details		tbd
* @note			The function will be modified on the HALO platform
*/
void SM_LLL_TimerDisable(SM_MotorId_t eMtrId)
{
  	STM_0.CHANNEL[eMtrId].CCR.R = 0x00000000u; 
}


/*==================================================================================================
                                       INTERRUPT HANDLERS
==================================================================================================*/

void _isr_SMC_Interrupt(void)
{
	uint8_t u8Copy;
	
	LLL_SMC_OVF_INT_CLEAR;			/* Clear SMC overflow interrupt flag */ 
	LLL_SMC_OVF_INT_DISABLE;		/* Disable SMC overflow interrupt */

	/* for SM_MAX_MOTOR times */
	for(u8Copy = 0; u8Copy < SM_MAX_MOTORS; u8Copy++)
	{
		/* If the corresponding motor has initialized HW */ 
		if(trSmcData[u8Copy].eMotorHwInit == SM_HW_MTR_INITIALIZED)
		{
			/* Update Sin duty cycle of the corresponding motor */
			RAL_RegWrite(trSmcData[u8Copy].u16MotorSin,&RAL_MOTOR_DUTY_SIN(u8Copy));
			/* Update Cos duty cycle of the corresponding motor */					
			RAL_RegWrite(trSmcData[u8Copy].u16MotorCos,&RAL_MOTOR_DUTY_COS(u8Copy));	
		}
	}					
}

/*================================================================================================*/
#ifdef __cplusplus
}
#endif